/** @file   framegrabber.h
 * @brief   Declarations of FrameGrabber - class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/01/29 20:17:48 $
 * @author  Tomi Lamminsaari
 */
#ifndef H_ENG2D_FRAMEGRABBER_H
#define H_ENG2D_FRAMEGRABBER_H

#include "eng2d_dll.h"
#include <string>

// Just let the compiler know about these classes
struct BITMAP;
struct RLE_SPRITE;


namespace eng2d {

// Forward declarations
class Size2D;


/** @class  FrameGrabber
 * @brief   Grabs smaller bitmaps from bigger bitmap.
 * @author  Tomi Lamminsaari
 *
 * This is a utility-class that can load and split larger
 * bitmap into a smaller bitmaps. This class is used when we load the
 * animation frames of the GameObjects.
 *
 * 
 */
class DLLIMPORT FrameGrabber
{
public:

  ///
  /// Constructors, destructor and operators
  /// ======================================
  
  /** Creates new FrameGrabber and assigns it to given bitmapfile.
   * @param     filename          name of the bitmapfile
   * @param     blockW            Width of the frames
   * @param     blockH            Height of the frames
   */
  FrameGrabber( const std::string& filename, int blockW, int blockH );
  
  /** Creates new FrameGrabber and assigns it to given bitmapfile.
   * @param     aFilename       Name of the bitmapfile.
   * @param     aGridSize       Size of the grid.
   */
  FrameGrabber( const std::string& aFilename, const Size2D& aGridSize );
  
  /** Destructor
   */
  virtual ~FrameGrabber();
  
  
  
  ///
  /// Methods
  /// =======
  
  /** Grabs and returns a frame from given position.
   * @param     col               From which frame column
   * @param     row               From which frame row
   * @return    Pointer to new bitmap that contains the graphics or
   *            null-pointer if grabbing fails. The caller is responssible
   *            of destroying this bitmap.
   */
  BITMAP* grab( int col, int row ) const;
  
  /** Grabs and converts the frame to rle-sprite format.
   * @param     col               From which column
   * @param     row               From which row
   * @return    Pointer to new rle-sprite that contains the frame
   *            graphics or null-pointer if fails.
   */
  RLE_SPRITE* grabRLE( int col, int row ) const;
  
protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Default constructor. This version is used by the child classes.
   */
  FrameGrabber();
  
  /** Reloads the bitmap.
   * @param     filename          Name of the bitmapfile.
   */
  void reload( const std::string& filename );
  
  
  ///
  /// Members
  /// =======
  
  /** Name of the bitmapfile */
  std::string m_filename;
  
  /** Width of the frames */
  int         m_blockWidth;
  
  /** Height of the frames */
  int         m_blockHeight;
  
  /** The bitmap itself where the frames can be found */
  BITMAP*     m_pBitmap;
};


};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: framegrabber.h,v $
 * Revision 1.2  2006/01/29 20:17:48  lamminsa
 * Constructor with Size2D parameter added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:38  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:19:43+02  lamminsa
 * Initial revision
 *
 */
